<?php

/*
 * ==========================================================
 * SLACK APP
 * ==========================================================
 *
 * Slack app main file
 *
 */

define('SB_SLACK', '1.0.0');

/*
 * -----------------------------------------------------------
 * SEND SLACK MESSAGE
 * -----------------------------------------------------------
 *
 * Send a message to Slack
 *
 */

function sb_send_slack_message($user_id, $full_name, $profile_image = SB_URL . '/media/user.svg', $message = '', $attachments = [], $channel = -1) {

    //Channel ID
    if (!sb_get_setting('slack-active')) {
        return ['status' => 'error', 'response' => 'Slack not active'];
    }
    $token = sb_get_setting('slack-token');
    if ($token == '') {
        return ['status' => 'error', 'response' => 'Slack token not found'];
    }
    $full_name = str_replace('#', '', $full_name);
    if ($channel == -1) {
        $channel = sb_get_slack_channel($user_id, $token);
        if ($channel[0] == -1 || $user_id == -1) {
            $channel = [sb_get_setting('slack-channel'), false];
        }
    } else {
        $channel = [$channel, false];
    }

    // Attachments
    $slack_attachments = '';
    for ($i = 0; $i < count($attachments); $i++) {
        $slack_attachments .= '{ "title": "' . $attachments[$i][0] . '", "title_link": "' .  $attachments[$i][1] . '" },';
    }

    // Send message to Slack
    $data = [ 'token' => $token, 'channel' => $channel[0], 'text' => $message, 'username' => $full_name, 'bot_id' => 'support-board', 'icon_url' => $profile_image, 'as_user' => false, 'attachments' => ($slack_attachments != '' ? '[' . substr($slack_attachments, 0, -1) . ']' : '')];
    $response = sb_curl('https://slack.com/api/chat.postMessage', $data, []);

    // Send a message to the main channel if the user's channel is new
    if ($channel[1] && isset($response['ok']) && $response['ok'] == true) {
        $user = sb_get_active_user();
        $user_extra = sb_get_user_extra($user_id);
        $channel_link = 'https://' . sb_get_setting('slack-workspace') . '.slack.com/archives/' . $channel[0];
        $data['channel'] = sb_get_setting('slack-channel');
        $data['text'] = '';

        // Fields
        $fields_include = ['location', 'browser', 'browser_language'];
        $fields = [['title' => sb_('Message'), 'value' => sb_db_sanatize($message), 'short' => false]];
        if (isset($user['email'])) {
            array_push($fields, ['title' => 'Email', 'value' => $user['email'], 'short' => true]);
        }
        if (is_array($user_extra)) {
            for ($i = 0; $i < count($user_extra); $i++) {
                if (in_array($user_extra[$i]['slug'], $fields_include)) {
                    array_push($fields, ['title' => $user_extra[$i]['name'], 'value' =>$user_extra[$i]['value'], 'short' => true]);
                }
            }
        }
        array_push($fields, ['title' => '', 'value' => '*<' . $channel_link . '|Reply in channel>*', 'short' => false]);
        $data['attachments'] = json_encode([['fallback' => 'A conversation started by' . ' ' . sb_db_sanatize($full_name) . '. *<' . $channel_link . '|Reply in channel>*"', 'text' => '*' . sb_('A conversation started by') . ' ' . sb_db_sanatize($full_name) . '*', 'color' => '#028be5', 'fields' => $fields]]);

        //Send message to slack
        $response = sb_curl('https://slack.com/api/chat.postMessage', $data, []);
    }
    if (isset($response['ok'])) {
        if ($response['ok']) {
            return ['success', $channel[0]];
        } else if (isset($response['error']) && $response['error'] == 'is_archived') {

            // Unarchive channel and send the message again
            $response = sb_curl('https://slack.com/api/channels.unarchive', ['token' => $token, 'channel' => $channel[0]], []);
            if ($response['ok']) {
                return sb_send_slack_message($user_id, $full_name, $profile_image, $message, $attachments);
            }
        }
    }
    return $response;
}

/*
 * -----------------------------------------------------------
 * GET SLACK CHANNEL
 * -----------------------------------------------------------
 *
 * Return the correct Slack channel of the user to send the messages to
 *
 */

function sb_get_slack_channel($user_id, $token) {
    $channels = sb_get_external_settings('slack-channels');
    if (isset($channels[$user_id])) {
        return [$channels[$user_id]['id'], false];
    } else {
        $active_user = sb_get_active_user();
        $username = str_replace('#', '', $active_user['first_name'] . '-' . $active_user['last_name']);
        $response = sb_curl('https://slack.com/api/channels.create', ['token' => $token, 'name' => $username], []);
        if (isset($response['channel'])) {
            $channels[$user_id] = ['id' => $response['channel']['id'], 'name' => $response['channel']['name']];
            $json = sb_json_enconde($channels);
            sb_db_query('INSERT INTO sb_settings(name, value) VALUES (\'slack-channels\', \'' . $json . '\') ON DUPLICATE KEY UPDATE value = \'' . $json . '\'');
            return [$response['channel']['id'], true];
        } else if (isset($response['error']) && $response['error'] === 'name_taken') {
            $channel_name = str_replace(' ', '-', strtolower($username));
            $response = sb_curl('https://slack.com/api/channels.list', ['token' => $token, 'exclude_archived' => true, 'exclude_members' => true], []);
            if (isset($response['channels'])) {
                $channel_id = -1;
                foreach ($response['channels'] as $channel) {
                    if ($channel['name'] == $channel_name) {
                        $channel_id = $channel['id'];
                        $channels[$user_id] = ['id' => $channel_id, 'name' => $channel['name']];
                        $json = sb_json_enconde($channels);
                        sb_db_query('INSERT INTO sb_settings(name, value) VALUES (\'slack-channels\', \'' . $json . '\') ON DUPLICATE KEY UPDATE value = \'' . $json . '\'');
                        break;
                    }
                }
                return [$channel_id, true];
            }
        }
    }
    return false;
}

/*
 * -----------------------------------------------------------
 * GET SLACK USERS
 * -----------------------------------------------------------
 *
 * Return the Slack users(agents) ID and name
 *
 */

function sb_slack_users() {
    $token = sb_get_setting('slack-token');
    $users = ['slack_users' => [], 'agents' => [], 'saved' => sb_get_setting('slack-agents')];
    if ($token != '') {
        $response = sb_curl('https://slack.com/api/users.list', ['token' => $token], []);
        if ($response['members']) {
            for ($i = 0; $i < count($response['members']); $i++) {
                $id =$response['members'][$i]['id'];
                $name = $response['members'][$i]['real_name'];
                if ($name != 'Slackbot') {
                    array_push($users['slack_users'], ['id' => $id, 'name' => $name]);
                }
            }
        }
        $agents = sb_db_get('SELECT id, first_name, last_name FROM sb_users WHERE user_type = "agent" OR user_type = "admin"', false);
        for ($i = 0; $i < count($agents); $i++) {
            array_push($users['agents'], ['id' => $agents[$i]['id'], 'name' => $agents[$i]['first_name'] . ' ' . $agents[$i]['last_name']]);
        }
        return $users;
    } else {
        return 'slack-token-not-found';
    }
}

/*
 * -----------------------------------------------------------
 * SLACK USERS STATUS CHECK
 * -----------------------------------------------------------
 *
 * Check if at least one Slack user is online
 *
 */

function sb_slack_agent_online($agent_id) {
    $slack_agent_id = -1;
    $token = sb_get_setting('slack-token');
    if ($token == '') {
        return 'offline';
    }
    $slack_agents = sb_get_setting('slack-agents');
    if ($slack_agents != false && !is_string($slack_agents)) {
        foreach ($slack_agents as $slack_id => $id) {
            if ($id == $agent_id) {
                $slack_agent_id = $slack_id;
                break;
            }
        }
    }
    if ($slack_agent_id == -1) {
        return 'offline';
    }
    $response = json_decode(file_get_contents('https://slack.com/api/users.getPresence?token=' . $token . '&user=' . $slack_agent_id), true);
    return (isset($response['ok']) && $response['presence'] == 'active') || (isset($response['online']) && $response['online'] === true) ? 'online' : 'offline';
}

/*
 * -----------------------------------------------------------
 * RENAME CHANNEL
 * -----------------------------------------------------------
 *
 * Rename a Slack channel
 *
 */

function sb_slack_rename_channel($user_id, $channel_name) {
    $channels = sb_get_external_settings('slack-channels');
    if (isset($channels[$user_id])) {
        $token = sb_get_setting('slack-token');
        if ($token != '') {
            $channel_name = str_replace(" ","-", strtolower($channel_name));
            $response = sb_curl('https://slack.com/api/channels.rename', ['token' => $token, 'channel' => $channels[$user_id]['id'], 'name' => $channel_name], []);
            if ($response['ok']) {
                $channels[$user_id]['name'] = $channel_name;
                $json = sb_json_enconde($channels);
                sb_db_query('INSERT INTO sb_settings(name, value) VALUES (\'slack-channels\', \'' . $json . '\') ON DUPLICATE KEY UPDATE value = \'' . $json . '\'');
                return true;
            }
        }
    }
    return false;
}

/*
 * -----------------------------------------------------------
 * ARCHIVE CHANNELS
 * -----------------------------------------------------------
 *
 * Archive all the Slack channel of the synchronized account
 *
 */

function sb_archive_slack_channels() {
    $token = sb_get_setting('slack-token');
    $response = sb_curl('https://slack.com/api/conversations.list', ['token' => $token, 'exclude_archived' => true, 'limit' => 999], []);
    if ($response['ok'] && isset($response['channels'])) {
        for ($i = 0; $i < count($response['channels']); $i++) {
            sb_curl('https://slack.com/api/conversations.archive', ['token' => $token, 'channel' => $response['channels'][$i]['id']], []);
        }
    } else {
        return new SBError('Error: ' . json_encode($response), 'sb_get_channels');
    }
    return true;
}

/*
 * -----------------------------------------------------------
 * [SLACK.PHP] LISTENER
 * -----------------------------------------------------------
 *
 * Receive and process the Slack messages of the Slack agents forwarded by board.support
 *
 */

function sb_slack_listener($response) {
    if (isset($response['event'])) {
        $response = $response['event'];
        $subtype = isset($response['subtype']) ? $response['subtype'] : '';

        // Message: Check if the json response is a valid message
        if (isset($response['type']) && $response['type'] == 'message' && $subtype != 'channel_join' && ($subtype == '' || $subtype == 'file_share') && ($response['text'] != '' || (is_array($response['files']) && count($response['files']) > 0))) {

            // Get the user id of the slack message
            $user_id = sb_slack_api_user_id($response['channel']);

            // Elaborate the Slack message
            if ($user_id != -1) {
                $last_message = sb_slack_api_last_message($user_id);
                $message = $response['text'];

                // Emoji
                $emoji = explode(':', $message);
                if (count($emoji)) {
                    $emoji_slack = json_decode(file_get_contents(SB_PATH . '/resources/json/emoji-slack.json'), true);
                    for ($i = 0; $i < count($emoji); $i++) {
                        if ($emoji[$i] != '') {
                            $emoji_code = ':' . $emoji[$i] . ':';
                            if (isset($emoji_slack[$emoji_code])) {
                                $message = str_replace($emoji_code, 'u' . $emoji_slack[$emoji_code], $message);
                            }
                        }
                    }
                }

                // Attachments
                $attachments = [];
                if ($subtype == 'file_share') {
                    $slack_files = $response['files'];
                    $token = sb_get_setting('slack-token');
                    for ($i = 0; $i < count($slack_files); $i++) {
                        $data = http_build_query($data);
                        $ch = curl_init('https://slack.com/api/files.sharedPublicURL');
                        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
                        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query(['token' => $token, 'file' => $slack_files[$i]['id']]));
                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
                        $attachment = json_decode(curl_exec($ch), true);
                        curl_close($ch);
                        if (isset($attachment['file'])) {
                            array_push($attachments, [$slack_files[$i]['title'], $attachment['file']['permalink_public']]);
                        }
                    }
                }

                // Set the user login for the security check
                global $SB_LOGIN;
                $SB_LOGIN = ['id' => $user_id,  'user_type' => 'user'];

                // Get the agent id
                $agent_id = sb_slack_api_agent_id($response['user']);

                // Send the message
                sb_send_message($agent_id, $last_message['conversation_id'], $message, $attachments, 1, $response);

                // Email notifications
                if (sb_get_setting('notify-user-email') && ($last_message['message_time'] == '' || strtotime("-5 minutes") > strtotime($last_message['message_time'])) && sb_get_user($user_id)['email'] != '') {
                    $agent = sb_db_get('SELECT first_name, last_name, profile_image FROM sb_users WHERE id = "' . $agent_id . '"');
                    sb_email_create($user_id, $agent['first_name'] . ' ' . $agent['last_name'], $agent['profile_image'], $message, $attachments);
                }
            }
        }

        // Event: message deleted
        if ($subtype == 'message_deleted') {
            $user_id = sb_slack_api_user_id($response['channel']);
            $agent_id = sb_slack_api_agent_id($response['previous_message']['user']);
            $last_message = sb_slack_api_last_message($user_id);
            $online = sb_update_users_last_activity(-1, $user_id) === 'online';
            $previous_message = $response['previous_message']['text'];
            sb_db_query(($online ? 'UPDATE sb_messages SET message = "", attachments = "", payload = "{ \"event\": \"delete-message\" }", creation_time = "' . gmdate('Y-m-d H:i:s') . '"' : 'DELETE FROM sb_messages') . ' WHERE (user_id = "' . $agent_id . '" OR user_id = "' . $user_id . '") AND conversation_id = "' . $last_message['conversation_id'] . '" AND ' . ($previous_message == '' ? 'attachments LIKE "%' . $response['previous_message']['attachments'][0]['title'] . '%"' : 'message = "' . $previous_message . '"') .' LIMIT 1');
        }

        // Event: message changed
        if ($subtype == 'message_changed') {
            $agent_id = sb_slack_api_agent_id($response['previous_message']['user']);
            sb_db_query('UPDATE sb_messages SET message = "' . sb_db_sanatize($response['message']['text']) . '", creation_time = "' . gmdate('Y-m-d H:i:s') . '" WHERE user_id = "' . $agent_id . '" AND payload LIKE "%' .  $response['message']['ts'] . '%" LIMIT 1');
        }
    }
}

/*
 * -----------------------------------------------------------
 * [SLACK.PHP] GET USER ID
 * -----------------------------------------------------------
 *
 * Get the user id of the slack message
 *
 */

function sb_slack_api_user_id($channel_id) {
    $user_id = -1;
    $channels = sb_get_external_settings('slack-channels');
    foreach ($channels as $key => $channel) {
        if ($channel['id'] == $channel_id) {
            $user_id = $key;
            break;
        }
    }
    return $user_id;
}

/*
 * -----------------------------------------------------------
 * [SLACK.PHP] GET AGENT ID
 * -----------------------------------------------------------
 *
 * Get the agent id
 *
 */

function sb_slack_api_agent_id($user) {
    $slack_agents = sb_get_setting('slack-agents');
    if (isset($slack_agents[$user])) {
        return $slack_agents[$user];
    } else {
        return sb_db_get('SELECT id FROM sb_users WHERE user_type = "admin" LIMIT 1')['id'];
    }
}

/*
 * -----------------------------------------------------------
 * [SLACK.PHP] LAST MESSAGE
 * -----------------------------------------------------------
 *
 * Get the last message sent by the user
 *
 */

function sb_slack_api_last_message($user_id) {
    $last_message = sb_db_get('SELECT conversation_id, creation_time FROM sb_messages WHERE user_id = "' . $user_id . '" ORDER BY creation_time DESC LIMIT 1');
    return [ 'conversation_id' => (isset($last_message['conversation_id']) ? $last_message['conversation_id'] : -1), 'message_time' => (isset($last_message['creation_time']) ? $last_message['creation_time'] : '')];
}

?>