<?php

/*
 * ==========================================================
 * DIALOGFLOW APP
 * ==========================================================
 *
 * Dialogflow app main file
 *
 */

define('SB_DIALOGFLOW', '1.0.0');

/*
 * -----------------------------------------------------------
 * SEND DIALOGFLOW BOT MESSAGE
 * -----------------------------------------------------------
 *
 * Send the user message to the bot and return the reply
 *
 */

$sb_recursion = true;
function sb_send_bot_message($conversation_id, $message, $token, $language = false) {
    $project_id = sb_get_setting('dialogflow-project-id');
    if (!sb_get_setting('dialogflow-active')) {
        return ['status' => 'error', 'response' => 'Dialogflow not active'];
    }
    if (!sb_get_setting('dialogflow-multilingual') || $language == false) {
        $language = ['en'];
    }

    // Retrive token
    if ($token == -1) {
        $token = sb_get_dialogflow_token();
        if (sb_is_error($token)) {
            return ['status' => 'error', 'response' => $token];
        }
    }

    // Knowledge Base
    $knowledges = [];
    if (sb_get_setting('dialogflow-knowledge-active')) {
        $names = sb_get_setting('dialogflow-knowledge');
        for ($i = 0; $i < count($names); $i++) {
            array_push($knowledges, 'projects/' . $project_id . '/knowledgeBases/' . $names[$i]['knowledge-name']);
        }
    }

    // Send user message to Dialogflow
    $query = ['queryInput' => ['text' => ['text' => $message, 'languageCode' => $language[0]]]];
    if (count($knowledges) > 0) {
        $query['queryParams'] = ['knowledgeBaseNames' => $knowledges];
    }
    $query = json_encode($query);
    $response = sb_curl('https://dialogflow.googleapis.com/v2beta1/projects/' . $project_id . '/agent/sessions/sb:detectIntent', $query, [ 'Content-Type: application/json', 'Authorization: Bearer ' . $token, 'Content-Length: ' . strlen($query) ]);
    $messages = isset($response['queryResult']) && isset($response['queryResult']['fulfillmentMessages']) ? $response['queryResult']['fulfillmentMessages'] : [];
    $results = [];

    // Dialogflow response
    for ($i = 0; $i < count($messages); $i++) {

        // Message
        $message = '';
        if (isset($messages[$i]['text'])) {
            $message = $messages[$i]['text']['text'][0];
            if (isset($messages[$i]['payload'])) {
                $i++;
            }
        } else if (isset($messages[$i]['simpleResponses']) && isset($messages[$i]['simpleResponses']['simpleResponses']) && isset($messages[$i]['simpleResponses']['simpleResponses']) && isset($messages[$i]['simpleResponses']['simpleResponses'][0]['textToSpeech'])) {
            // Google Assistant Message
            $message = $messages[$i]['simpleResponses']['simpleResponses'][0]['textToSpeech'];
        }

        // Attachments
        $attachments = [];
        if (isset($messages[$i]['payload'])) {
            if (isset($messages[$i]['payload']['attachments'])) {
                $attachments = $messages[$i]['payload']['attachments'];
                if ($attachments == '' && !is_array($attachments)) {
                    $attachments = [];
                }
            }
        }

        // Send the bot message to Support Board
        if ($message != '' || count($attachments)) {
            $bot_id = sb_db_get('SELECT id FROM sb_users WHERE user_type = "bot" LIMIT 1');
            if (isset($bot_id['id'])) {
                $status = sb_get_setting('bot-unknow-notify') && isset($response['response']['queryResult']) && $response['response']['queryResult']['action'] != 'input.unknown' ? 1 : -1;
                $response_db = sb_send_message($bot_id['id'], $conversation_id, $message, $attachments, $status, $response);
                if ($response_db == true) {
                    array_push($results, ['message' => $message, 'attachments' => $attachments]);
                }
            }
        }
    }

    if (count($results)) {

        // Return the bot messages list
        return ['status' => 'success-bot', 'token' => $token, 'messages' => $results, 'response' => $response];
    } else if (isset($response['error']) && $response['error']['code'] == 401){

        // Reload the function and force it to generate a new token
        global $sb_recursion;
        if ($sb_recursion) {
            $sb_recursion = false;
            return sb_send_bot_message($conversation_id, $message, -1, $language);
        }
    }
    return ['status' => 'error', 'response' => $response];
}

/*
 * -----------------------------------------------------------
 * GET TOKEN
 * -----------------------------------------------------------
 *
 * Get a fresh Dialogflow access token
 *
 */

function sb_get_dialogflow_token() {
    $token = sb_get_setting('dialogflow-token');
    if ($token == '' || $token === false) {
        return new SBError('Dialogflow Refresh Token not found', 'sb_get_dialogflow_token');
    }
    $response = file_get_contents('https://board.support/synch/dialogflow.php?refresh-roken=' . $token);
    if ($response != 'api-dialogflow-error' && $response != false) {
        $token = json_decode($response, true);
        if (isset($token['access_token'])) {
            return $token['access_token'];
        }
    }
    return new SBError('Dialogflow Refresh Token error. Response: ' . $response, 'sb_get_dialogflow_token');
}

/*
 * -----------------------------------------------------------
 * CREATE INTENT
 * -----------------------------------------------------------
 *
 * Create a new Dialogflow intent
 *
 */

function sb_dialogflow_intent($expressions, $bot_response, $language = '') {
    if (!sb_get_setting('dialogflow-active')) {
        return ['error' => true, 'message' => 'Dialogflow not active.'];
    }

    // Retrive token
    $token = sb_get_dialogflow_token();
    if (sb_is_error($token)) {
        return ['status' => 'error', 'response' => $token];
    }

    // Send user message to Dialogflow
    $training_phrases = [];
    for ($i = 0; $i < count($expressions); $i++) {
        array_push($training_phrases, [ 'type' => 'TYPE_UNSPECIFIED', 'parts' => [ 'text' => $expressions[$i]]]);
    }
    if ($language != '') {
        $language = '?languageCode=' . $language;
    }
    $project_id = sb_get_setting('dialogflow-project-id');
    $query = json_encode(['displayName' => $expressions[0], 'priority' => 500000, 'webhookState' => 'WEBHOOK_STATE_UNSPECIFIED', 'trainingPhrases' => $training_phrases,  'messages' => [['text' => ['text' => $bot_response]]]]);
    $response = sb_curl('https://dialogflow.googleapis.com/v2/projects/' . $project_id . '/agent/intents' . $language, $query, [ 'Content-Type: application/json', 'Authorization: Bearer ' . $token, 'Content-Length: ' . strlen($query) ]);
    if (isset($response['displayName'])) {
        return true;
    }
    return $response;
}

?>